#!/bin/bash
#
# make_snmp_conf.sh
#
# version 3 11/30/2017 BLL 
#
#
# start time
started_at=$(date +'%Y-%m-%d-%H-%M-%S')
# Determine whether bulk or one-off:
# Bulk will have a single parameter passed in that is a file name.
# One-off will have no parameters passed in but will require answers for each parameter.
#
# Test for one parameter passed in
# 
if [ $# -eq 1 ]; then
  mode='bulk'
else
  mode='one'
fi
echo "mode is: $mode"

# The parameters showing on the original equipment spreadsheet are below.
# They are needed for a complete snmpd.conf file to be used by zabbix.
# The originals are as follows:
#
#  1.	Business Group 
#  2.	Region		
#  3.	Sitecode
#  4.	Site Description
#  5.	Host Name
#  6.	DNS Domain
#  7.	Group
#  8.	Hardware Model Name
#  9.	Age
# 10.	Last Check-in Date
# 11.	Last Kickstart Date
# 12.	Kickstart Count
# 13.	Last Boot Update
# 14.	Kernel Version
# 15.	OS Release
# 16.	Last YUM Update
# 17.	Estimated Manufacturing Date
# 18.	Serial Number
# 19.	CPU Manufacturer
# 20.	CPU Vendor
# 21.	CPU Frequency
# 22.	CPU Family
# 23.	Memory RAM Total
# 24.	Disk System Disk Size
# 25.	Disk Type
# 26.	RAM Speed
# 27.	RAM Type
# 28.	GPU Card
# 29.	Satellite Registration Status
# 30.	System ID
# 31.	Last Local Users
# 32.	CPU Sockets
# 33.	CPU Physical Cores
# 34.	CPU Total Cores
# 35.	IP Addresses
# 36.	Client IP Address
# 37.	Memory GB
# 38.	Servers
# 39.	Workstations
# 40.	VM
# 41	Unknown
# 42.	Age
# 43.	 < 3 yrs
# 44.	 < 5 yrs
# 45.	 5 to 10 yrs
# 46.	 > 10 yrs	
if [ "$mode" == "one" ]; then
  # this feature comes with 3.1 version
  echo "This begins the one-off Q and A"
else 
  # bulk uses the spreadsheet info to build snmp.conf sets unique by region, site, and sysDesc parameters
  # this uses PDS committee's suggestions as follows:
  # Inventory, satellite and SNMPD (Zabbix, etc) will use one or more of the following 7 keyword tags to categorize computer:
  #
  # select which mode to run this script. bulk has an input file and otherwise run a one-off
  if [ "$mode" == "bulk" ]; then
    #
    # see if a good file name was fed in at the command line
    if [ -e "$1" ]; then
      #
      # pass the file name to use into a parameter
      in_file=$1

      # 
      # check for the working subdir
      if [ ! -d out ]; then
          mkdir out
      else
          # if it is clean it out
          rm -rf out/*
      fi

      # 
      # check for the undefined file
      if [ -f "out/define_utility.txt" ]; then
        # if it is clean it out
        rm -rf out/define_utility.txt
      fi

      # 
      # check for the undefined file
      if [ -f "*undefined.snmp.conf" ]; then
        # if it is clean it out
        rm -rf out/*undefined.snmp.conf
      fi
      
      #
      # check for a RESULTS_README file
      if [ -f "out/RESULTS_README" ]; then
        # if it is clean it out
        rm -rf out/RESULTS_README
      fi



      #
      # Do a while loop through the file one line at a time to separate
      # out each parameter set and write the unique snmp.conf file for a region, site, machine type, and business unit.
      # Within this loop determine if an exact unique snmp.conf exists and skip on to the next record.
      # set the created record counter
      rec_ctr=0
      err1_ctr=0
      err2_ctr=0 
      # this while looks at the tabs between the columns.
      # be sure that the input file is extracted correctly from the Excell spreadsheet
      while IFS=$'\t' read v1 v2 v3 v4 v5 v6 v7 v8 v9 v10 v11 v12 v13 v14 v15 v16 v17 v18 v19 v20 v21 v22 v23 v24 v25 v26 v27 v28 v29 v30 v31 v32 v33 v34 v35 v36 v37 v38 v39 v40 v41 v42 v43 v44 v45 v46 v47; do
	# 
	# Display the file content line individual variables
	echo "------- Record Number -------: $rec_ctr"
        echo "               Businees Group: $v1"
        echo "                       Region: $v2"
   	echo "                    Site Code: $v3"
	echo "             Site Description: $v4"
	echo "                    Host Name: $v5"
	echo "                   DNS Domain: $v6"
	echo "                        Group: $v7"
	echo "          Hardware Model Name: $v8"
	echo "                          Age: $v9"
	echo "           Last Check-In Date: $v10"
	echo "          Last Kickstart Date: $v11"
	echo "              Kickstart Count: $v12"
	echo "             Last Boot Update: $v13"
	echo "               Kernel Version: $v14"
	echo "                   OS Release: $v15"
	echo "              Last YUM Update: $v16"
	echo " Estimated Manufacturing Date: $v17"
	echo "                Serial Number: $v18"
	echo "             CPU Manufacrurer: $v19"
	echo "                   CPU Vendor: $v20"
	echo "                CPU Frequency: $v21"
	echo "                   CPU Family: $v22"
	echo "             Memory RAM Total: $v23"
	echo "        Disk System Disk Size: $v24"
	echo "                    Disk Type: $v25"
	echo "                    RAM Speed: $v26"
	echo "                     RAM Type: $v27"
	echo "                     GPU Card: $v28"
	echo "Satellite Registration Status: $v29"
	echo "                    System ID: $v30"
	echo "             Last Local Users: $v31"
	echo "                  CPU Sockets: $v32"
	echo "           CPU Physical Cores: $v33"
	echo "                 IP Addresses: $v35"
	echo "            Client IP Address: $v36"
	echo "              Memory Gigabyte: $v37"
	echo "                      Servers: $v38"
	echo "                 Workstations: $v39"
	echo "                           VM: $v40"
	echo "                      Unknown: $v41"
        echo "                          Age: $v42"
	echo "                    < 3 years: $v43"
	echo "                    < 5 years: $v44"
	echo "                5 to 10 years: $v45"
        echo "                   > 10 years: $v46"
        echo "                Any Remainder: $v47"
	echo
  	#
  	## utility group (not used as keyword) 
  	# The machine naming convention for the machines primary function follows GIS ID standard: ${sitecode}LSxxx (ea. czbr01ls001)
  	#	OS = RHEL Server
  	#	infrastructure - servers running infrastructure services like: NIS, DNS, CUPS, SQL, APACHE, BACKUP, SVN, TRAC, WIKI,.
  	#	hypervisor - KVM, VMWare
  	#	lab - non-production testing machine of any type
  	#	session - VNC/EoD/ETX/NX servers
  	#
  	## workstation group - single user machine
  	# The machine naming convention of its primary function follows GIS ID standard: ${sitecode}LWxxx (ea. czbr01lw001)
  	#	OS = RHEL Workstation
  	# 
  	## compute group any compute server
  	# The machine naming convention of its primary function follows GIS ID standard: ${sitecode}LCxxx (ea. czbr01lc001)
  	#	OS = RHEL Server
  	#
  	## special case group - undefined machine 
  	# The machine naming convention for these machines will be to set a default tag in snmpd.conf so we can search and find machines without tags set.
  	#
  	## Tagging Examples
  	# 
  	# Mixed role servers will have multiple tags.
  	# Example: utility machine in Bucharest running NIS slave, print server and VNCs will be tagged as: infrastructure,session
  	#
  	# Additionally we will put and keep Business Unit information in this tag sequence.  This will allow us to propagate this to any inventory tool:
  	# BU keywords:
  	#	R&D
  	#	ISG
  	#	APG
  	#
  	# The example format will look like this:
  	#	sysDescr TAGS:comma,separated,tags BU:COMMA,SEPARATED,BU,KEYWORDS
  	#
  	# The default format will look like this
  	#	sysDescr TAGS:undefined BU:undefined
  	#
	
	
  	#
  	# So to begin set the listed values into TAG and BU or set the defaults into TAGS and BU
  	#
  	# parse the string of the hostname to determine the machine type
  	# do it by taking the sitecode off the name and the next two letters tell what tag to use
  	# hostname 
  	
  	# use the hostname 
  	str_2_parse=$v5
  	# sitename to remove from hostname
  	str_2_remove=$v3
  	# measure the sitename length
  	length_of_sitename=${#str_2_remove}
  	# make a shorten name with the sitename removed
  	shortened_name=${str_2_parse:$length_of_sitename}
  	# look at the first two characters of the shortened name
  	if [ ${shortened_name:0:2} == "ls" ] || [ ${shortened_name:0:2} == "lw" ] || [ ${shortened_name:0:2} == "lc" ]; then
	
  	  # utility should have additional information on type field like infrastructure, hypervisor, lab, or session.
  	  # All we have in the file is utility for ls named machines.  The default is TAG:undefined.
	
  	  # utility
  	  if [ ${shortened_name:0:2} == "ls" ]; then
            # use the name in the group field if it exists
            if [ ! "$v7" == "" ]; then
              if [ "$v7" == "utility" ]; then
                tag="TAG:undefined "$v7",infrastructure,hypervisor,lab,session"
              fi
              if [[ "$v7" == *"infrastructure"* ]] || [[ "$v7" == *"hypervisor"* ]] || [[ "$v7" == *"lab"* ]] || [[ "$v7" == *"session"* ]]; then
                tag="TAG:"$v7
              fi
            else
	      tag="TAG:undefined"
            fi
  	  fi
  	  # workstation
  	  if [ ${shortened_name:0:2} == "lw" ]; then
		tag="TAG:workstation"
  	  fi
  	  # compute
  	  if [ ${shortened_name:0:2} == "lc" ]; then
		tag="TAG:compute"
  	  fi
	
  	else

	# Default undefined
		tag="TAG:undefined"
  	fi # if [ ${shortened_name:0:2} == "ls" ] || [ ${shortened_name:0:2} == "lw" ] || [ ${shortened_name:0:2} == "lc" ]: then

	#
	# the business unit is clearly stated in the first field
	# if it is not a blank or null
	if [ ! "$v1" == "" ]; then

	  # R&D
          if [ "$v1" == "R&D" ] || [ "$v1" == "r&d" ]; then
	    bu="BU:R&D"
          fi
          # ISG
          if [ "$v1" == "ISG" ] || [ "$v1" == "isg" ]; then
	    bu="BU:ISG"
          fi
          # APG
          if [ "$v1" == "APG" ] || [ "$v1" == "apg" ]; then
	    bu="BU:APG"
          fi
        else
          # Default undefined
          bu="BU:undefined"
        fi

	# make snmp file name to produce
	outfile="out/"$v2"."$v3"."$tag"."$bu".snmp.conf"
        undefined="out/"$v2"."$v3"."$v5"."$v7"."$tag"."$bu".undefined.snmp.conf"
        define_utility="out/"$v2"."$v3"."$v5".utility."$bu".define_utility.snmp.conf"
	# show em
        echo "Output File: "$outfile	
        # check if it does not already exist and make one otherwise skip this record
	if [ ! -f "$outfile" ]; then
	  #
	  # make a file for the machine's snmpd.conf
  	  #
	  # 1st line
	  echo "# PDS SNMP configuration file for Region "$v2" and Sitecode "$v3"    Created on "$(date -u +'%Y-%m-%d') > $outfile
	  # second line
	  echo "#" >> $outfile 
	  # third line
	  echo "# community	view name	IP/Hostname (PTR, A, or CNAME)	Description" >> $outfile
	  # fourth line
	  echo "#--------------	--------------- ------------------------------- -----------------------------------------------------------------" >> $outfile
	  # fifth thru tenth lines
	  echo "rocommunity	public		10.253.110.130			# dawson.onsemi.com" >> $outfile
	  echo "rocommunity 	onsemi-view	pds-zabbix-"$v2".onsemi.com	# CNAME for regional zabbix server" >> $outfile
	  echo "rocommunity 	onsemi-view	10.222.65.150			# seahawk.onsemi.com 			APAC SolarWinds" >> $outfile
	  echo "rocommunity 	onsemi-view	10.222.0.150			# chinook.onsemi.com 			NA SolarWinds" >> $outfile
	  echo "rocommunity 	onsemi-view	10.222.128.150			# jetranger.onsemi.com			EMEA SolarWinds" >> $outfile
	  echo "rocommunity 	onsemi-view	10.100.16.121  			# flash.onsemi.com			test machine for snmpwalk" >> $outfile
   	  #
 	  # site based decisions made from here down
	  # script logic is based on original spreadsheet and contents of columns
	  #
	  ##Site specific setup:
 	  #
 	  # site admin machine
	  echo "#" >> $outfile
	  echo "# Specify admin machine here" >> $outfile
	  echo "# community	view-name	IP/Hostname			Description" >> $outfile
	  echo "#-------------- --------------- ------------------------------- -----------------------------------------------------------------" >> $outfile
	  echo "rocommunity	onsemi-view	"$v3"adm.onsemi.com		# CNAME for site admin machine" >> $outfile
 	  #
	  # site code
	  echo "#" >> $outfile
	  echo "# sysLocation	name (quote enclosed)" >> $outfile
	  echo "#-------------- ---------------------" >> $outfile
	  echo "sysLocation 	"$v3 >> $outfile
	  #
	  # system contact e-mail list/group
	  echo "#"
	  echo "# sysContact	PDS Group E-Mail Address" >> $outfile
	  echo "#-------------- ------------------------" >> $outfile
	  echo "sysContact  PDS-"$v2"@onsemi.com" >> $outfile
	  echo "#" >> $outfile 
	  echo "# System Description"
	  # modify to only have system type recorded in snmp.conf file hereafter BLL 11/28/2017
	  # echo all of it to visually verify
	  echo "sysDescr "$tag","$bu
	  # this series adds the uses to the prefix of the sysDescr and output to snmp.conf for this system
          echo "sysDescr "$tag","$bu >> $outfile
          echo "#"

          # this is for output direction to correct files
          # two kinds of errors plain undefined and utility without further definition
          if [[ "$tag" == *"undefined"* ]] || [[ "$bu" == *"undefined"* ]]; then
            if [[ "$tag" == *",infrastructure,hypervisor,lab,session"* ]]; then
              echo $define_utility >> out/define_utility.txt
              (( err_ctr1++ ))
              
            else
              echo $undefined >> out/undefined.txt
              (( err_ctr2++ ))
            fi
          fi

  	  # show the finished file
	  cat $outfile
	  # for debug
	  #sleep 5
          # increase the record counter as we just made one new one
          (( rec_ctr++ ))
          # end of test for the file not existing
        fi # if [ ! -f "$outfile" ]; then
      done < $in_file # end of while loop
      #
      # make a results file
      ended_at=$(date +'%Y-%m-%d-%H-%M-%S')
      echo "Ran "$0" using input file "$1 >> out/RESULTS_README
      echo "Started at: "$started_at" and Ended at: "$ended_at >> out/RESULTS_README
      echo "It produced "$rec_ctr" unique SNMP.CONF files" >> out/RESULTS_README
      echo "It produced "$err_ctr1" error SNMP.CONF files. i.e.: sysDescr:utility without one of infrastructure,hypervisor,lab, or session " >> out/RESULTS_README
      echo "It produced "$err_ctr2" error SNMP.CONF files. i.e.: sysDescr:undefined i.e.: group mismatched to file name, bad hostnaming, etc. " >> out/RESULTS_README
    else
      # called for bulk but did not put a good file name in
      echo
      echo "You entered $0 $1 at the command prompt"
      echo 
      echo "The input file $1 cannot be found. Please check the name and try again..."
      echo
    fi # if [ -e "$1" ]; then
  fi # if [ "$mode" == "bulk" ]; then
fi # if [ "$mode" == "one" ]; then
exit 0 





