#!/bin/bash

# digg.sh
#  Usage: digg [domain|ip|url] [afrinic|apnic|arin|lacnic|ripe]
#  Examples:
#       digg https://google.com/ arin
#       digg 67.1.1.1 arin
#       digg www.example.com arin



# Fuction: Test an IP address for validity:
# Usage:
#      valid_ip IP_ADDRESS
#      if [[ $? -eq 0 ]]; then echo good; else echo bad; fi
#   OR
#      if valid_ip IP_ADDRESS; then echo good; else echo bad; fi
#
function valid_ip()
{
    local  ip=$1
    local  stat=1

    if [[ $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
        OIFS=$IFS
        IFS='.'
        ip=($ip)
        IFS=$OIFS
        [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 \
            && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]]
        stat=$?
    fi
    return $stat
}
#
#
# functions for each internet name registry
# restful apis differ by name registrar
#
# 
function afrinic()
{
  curl "https://rest.db.afrinic.net/ripe/ip/${1}.txt"
}

function lacnic()
{
  curl "https://rest.db.lacnic.net/ripe/ip/${1}.txt"
}

function ripe()
{
  curl "https://rest.db.ripe.net/ripe/ip/${1}.txt"
}

function arin()
{
  curl "http://whois.arin.net/rest/ip/${1}.txt"
}

function apnic()
{
  curl "http://whois.apnic.net/rest/ip/${1}.txt"
}

# If this is an IP, just get the name registrar Whois info
if valid_ip "$1"; then
  if [ "$2"=="arin" ]; then
    arin "$1"
  fi
  if [ "$2"=="apnic" ]; then
    apnic "$1"
  fi
  if [ "$2"=="ripe" ]; then
    ripe "$1"
  fi
  if [ "$2"=="arin" ]; then
    afrinic "$1"
  fi
  if [ "$2"=="arin" ]; then
    lacnic "$1"
  fi


else
  # extract the protocol
  proto=$(echo "$1" | grep :// | sed -e's,^\(.*://\).*,\1,g')
  # remove the protocol
  url=$(echo "${1/$proto/}")
  # extract the user (if any)
  user=$(echo "$url" | grep @ | cut -d@ -f1)
  # extract the host
  host="$(echo "${url/$user@/}" | cut -d/ -f1)"
  base_domain=$(sed -E 's/.*\.([a-z]+\.[a-z]+$)/\1/' <<< "$host")
  # by request - try to extract the port
  #port="$(echo $host | sed -e 's,^.*:,:,g' -e 's,.*:\([0-9]*\).*,\1,g' -e 's,[^0-9],,g')"
  # extract the path (if any)
  #path="$(echo $url | grep / | cut -d/ -f2-)"

  echo "--------------------------------------"
  echo "----------------- Whois --------------"
  echo "--------------------------------------"
  whois "$base_domain" | grep 'Registr\|Name\|Date' # Only look for important info

  echo "--------------------------------------"
  echo "------------ Name Servers ------------"
  echo "--------------------------------------"
  dig "$base_domain" -t NS +noall +answer

  echo ""
  echo "--------------------------------------"
  echo "----------------- MX -----------------"
  echo "--------------------------------------"
  dig "$base_domain" -t MX +noall +answer

  echo ""
  echo "--------------------------------------"
  echo "---------- A(AAA), CNAME -------------"
  echo "--------------------------------------"
  dig "$host" +noall +answer
  dig "$base_domain" +noall +answer

  echo ""
  echo "--------------------------------------"
  echo "----------- ARIN WHOIS ---------------"
  echo "--------------------------------------"
  for ip in $(dig "$host" +short); do
    if valid_ip "$ip"; then
      if [ "$2"=="arin" ]; then
        arin "$1"
      fi
      if [ "$2"=="apnic" ]; then
        apnic "$1"
      fi
      if [ "$2"=="ripe" ]; then
        ripe "$1"
      fi
      if [ "$2"=="arin" ]; then
        afrinic "$1"
      fi
      if [ "$2"=="arin" ]; then
        lacnic "$1"
      fi
    fi
  done
fi
